// ***********************************************************************************
// Programs written by R.Swan - rs108@mdx.ac.uk - www.netyaroze-europe.com/~middex2
// Taken from the tutorial by P.Passmore - p.passmore@mdx.ac.uk
//   Tutorial file can be found at the above web site
// Step ten - add a road to drive around on
// ***********************************************************************************

// ***********************************************************************************
// Preprocessor functions
// ***********************************************************************************

// **** include libraries and files
#include <stdio.h>                                    // load standard libraries
#include <libps.h>                                    // load playstation library
#include "pad.h"                                      // load joypad reading

// **** set up defines
#define true          (1)
#define false         (0)
#define OTABLE_LENGTH (12)                            // define length of OTable
#define MAX_PACKETS   (248000)                        // define size of GPU scratchpad
#define IS_PAL_USED   (true)                          // true = PAL mode, false = NTSC mode
#define PAL_WIDTH     (320)                           // size of PAL screen
#define PAL_HEIGHT    (240)
#define NTSC_WIDTH    (320)                           // size of NTSC screen
#define NTSC_HEIGHT   (256)

#define GROUND_MAXX   (15)                            // define ground x size
#define GROUND_MAXZ   (15)                            // define ground z size
#define GROUND_MAX_OBJECTS (225)  // total max number of ground objects
#define GROUND_SIZE   (1200)                          // X and Z dimensions of one road object

#define CAR_MEM_ADDR      (0x80090000)                // location of car .TMD data
#define ROAD_MEM_ADDR     (0x80098000)                // location of track .TMD data
#define ROAD_TEX_MEM_ADDR (0x800B0000)                // location of texture .TIM data

// **** create global variables, structures, headers and arrays (mostly system stuff)
int         Running = true;                           // define program to be running
u_long      VerticalSync = 0;                         // set up speed counter for processor

GsOT        OTable_Header[2];                         // Header files holding OTable info
GsOT_TAG    OTable_Array[2][1<<OTABLE_LENGTH];        // Array to hold drawing order info
PACKET      Packet_Memory[2][MAX_PACKETS];            // Memory to hold GPU packet data

typedef struct                                        // create structure called
  {                                                   // Track_Header0
  long            numberObjects;                      //   integer holding number of models
  GsDOBJ2         Object_Handler[GROUND_MAX_OBJECTS]; //   contains model info
  GsCOORDINATE2   Object_Coord[GROUND_MAX_OBJECTS];   //   contains object location and
                                                      //   orientation info
  u_long          *Object_Pointer[GROUND_MAX_OBJECTS];//   array of pointers to TMD data
  } Track_Header0;
Track_Header0     TrackData;                          // create instance of structure for use

typedef struct                                        // create structure called
  {                                                   // Object_Header0
  SVECTOR         rotation;                           //   vector array to hold rotation
  long            speed;                              //   integer holding velocity of car
  GsDOBJ2         Object_Handler;                     //   contains model info
  GsCOORDINATE2   Object_Coord;                       //   contains object location and
                                                      //   orientation info
  } Object_Header0;
Object_Header0    CarModel;                           // create instance of structure for car

GsFOGPARAM  Fogging;
GsF_LIGHT   LightSource;                              // Array of three light sources
GsRVIEW2    Camera[8];                                // array of different available cameras
long        cameraUsed = 0;                           // which camera number is in use
long        cameraCounter = 0;                        // holds time until new camera chosen

char RawGroundArray[GROUND_MAXX][GROUND_MAXZ] ={      // array holding ground data
{'1','1','1','1','1','1','1','0','0','0','1','1','1','1','1'},
{'0','0','0','0','1','0','1','1','1','1','1','0','0','0','1'},
{'0','0','0','0','1','0','0','0','0','0','1','0','0','0','1'},
{'0','0','0','0','1','0','0','0','0','0','1','1','1','1','1'},
{'0','0','0','0','1','0','0','0','0','0','0','0','1','0','0'},
{'1','1','1','1','1','1','1','0','0','0','0','0','1','0','0'},
{'1','0','0','0','0','0','1','0','0','0','0','0','1','0','0'},
{'1','0','0','0','0','0','1','0','0','0','0','0','1','0','0'},
{'1','0','0','1','1','1','1','1','1','1','1','1','1','1','1'},
{'1','0','0','1','0','0','0','0','0','0','0','0','0','0','1'},
{'1','0','0','1','0','0','0','0','0','0','0','0','0','0','1'},
{'1','0','0','1','0','0','1','1','1','1','1','1','1','1','1'},
{'1','0','0','1','0','0','1','1','1','1','0','0','0','0','0'},
{'1','0','0','1','0','0','1','1','1','1','0','0','0','0','0'},
{'1','1','1','1','0','0','1','1','1','1','0','0','0','0','0'}};

// **** function prototyping
void ProcessUserInput();                              // react to joypad use
void LoadTexture(long tMemAddress);                   // load a texture into video memory
void InitModel(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress);
                                                      // general object initialisation function
void InitTrack();                                     // set up the track
void AddModelToHeader(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress);
                                                      // join a 3d model with it's info
void AddModelToTrack(Track_Header0 *tTrack, int tX, int tY, int tZ, u_long *tMemAddress);
                                                      // join a 3d model to track info
void InitAllLights();                                 // define all lights to be used
void InitLight(GsF_LIGHT *tLight, int tNum, int tXv, int tYv, int tZv, int tR, int tG, int tB);
                                                      // set information for a lightsource
void InitView(GsRVIEW2 *tView, int tProjDist, int tRZ,// define details of scene viewpoint
                  int tFromX, int tFromY, int tFromZ,
                  int tToX, int tToY, int tToZ,
                  GsCOORDINATE2 *tReference);
void InitialiseGraphics();                            // define graphics initialisation
void RenderWorld();                                   // define how to draw the screen
void DrawObject(Object_Header0 *tObject, GsOT *tOTable);
                                                      // send a draw object command to queue
void DrawTrack(Track_Header0 *tTrack, GsOT *tOTable); // send a draw complete track command
void MoveModelTo(Object_Header0 *tObject, int tXc, int tYc, int tZc);
                                                      // move object to (tXc, tXy, tZc)
void MoveModelBy(Object_Header0 *tObject, int tXv, int tYv, int tZv);
                                                      // translate an object by a vector
void MoveModelForward(Object_Header0 *tObject, int tSpeed);
                                                      // move object in direction it faces
void RotateModel(Object_Header0 *tObject, int tXr, int tYr, int tZr);
                                                      // rotate an object (4096 = 360 degrees)
void ResetRotation(short tArray[3][3]);               // reset rotation part of a matrix
void SelectCamera(long tNumber);                      // routine to set the camera

// ***********************************************************************************
// Main function
// ***********************************************************************************

void main()
  {
  FntLoad(960, 256);                                  // load font into area of frame
                                                      // specified by 960, 256
  FntOpen(-160, -120, 320, 100, 0, 512);              // specify where on screen text goes
                                                      // (0, 0) is center of screen
  PadInit();                                          // initialise joypad reading
  InitialiseGraphics();                               // initialise graphics
  InitView(&Camera[0], 250, 0, 0,-1500,-1000, 0,0,0 ,false);
  InitView(&Camera[1], 250, 0, 0,-500,-1500, 0,-250,-500 ,&CarModel.Object_Coord);
  InitView(&Camera[2], 250, 0, 0,-1000, -1000, 0,-250,0 ,false);
  InitView(&Camera[3], 250, 368640, 0,-5000,2500, 0, 0, 2500 ,&CarModel.Object_Coord);
  InitView(&Camera[4], 250, 0, 0,-270,-400, 0,-245,0 ,&CarModel.Object_Coord);
  InitView(&Camera[5], 250, 0, 8400,-15000,8400, 0,0,0 ,false);
  InitView(&Camera[6], 250, 40960, -400,-200,-300, 0,0,500 ,&CarModel.Object_Coord);

  InitAllLights();                                    // call function that creates two lights
  InitTrack();
  InitModel(&CarModel, 0, 0, 0, (u_long *)CAR_MEM_ADDR);
                                                      // prepare objects for use
                                                      // car's position set to (0, -200, 0);

  Fogging.dqa = -5000;                                // this is a measure of the distance
                                                      // at which objects start to fade to
                                                      // the specified colours. IT IS NOT
                                                      // directly related to world units.
                                                      // The more negative the value, the
                                                      // farther in front of you an object
                                                      // has to be before it starts to fade
  Fogging.dqb = 5120*5120*0.8;                        // this is a measure of how far either
                                                      // side of the distance above that
                                                      // objects fade in or out. A value of
                                                      //   5120*5120*2   = very abrupt fading
                                                      //   5120*5120*0.7 = very gradual fade
  Fogging.rfc = 0;                                    // set colour for objects to fade to
  Fogging.gfc = 0;
  Fogging.bfc = 0;
  GsSetLightMode(1);                                  // set fogging to be used in light
  GsSetFogParam(&Fogging);                            // define which fogging to be used

  while(Running)                                      // do loop while program is running
    {
    if (cameraCounter >0)                             // if automatic camera choosing is on...
      {
      FntPrint("Automatic camera\n");                 // ...then print just that
      cameraCounter--;                                // decrement the counter
      if (cameraCounter == 0)                         // when it reaches zero...
        {
        cameraUsed = (cameraUsed+1) % 7;              // ...choose next camera
        SelectCamera(cameraUsed);                     // select the camera and fogging info
        cameraCounter = 150;                          // reset counter
        };
      };
    switch (cameraUsed)                               // update a camera's viewing info
      {
      case 0    : {                                   // static camera
                  long tScale = 9 * GROUND_SIZE;
                  long tScale2 = (tScale-1) / 2;
                  FntPrint("Static camera\n");
                  Camera[0].vpx = ((CarModel.Object_Coord.coord.t[0]) / tScale * tScale) + tScale2;
                  Camera[0].vpz = ((CarModel.Object_Coord.coord.t[2]) / tScale * tScale) + tScale2;
                  Camera[0].vrx = CarModel.Object_Coord.coord.t[0];
                  Camera[0].vry = CarModel.Object_Coord.coord.t[1];
                  Camera[0].vrz = CarModel.Object_Coord.coord.t[2]; break;};
      case 1    : {                                   // tracking camera
                  FntPrint("Tracking camera\n"); break;};
      case 2    : {                                   // scrolling side camera
                  FntPrint("Side camera\n");
                  Camera[2].vpx = CarModel.Object_Coord.coord.t[0];
                  Camera[2].vpy = -1000 - (abs(CarModel.Object_Coord.coord.t[2]) / 2);
                  Camera[2].vpz = -1000 + CarModel.Object_Coord.coord.t[2] / 4;
                  Camera[2].vrx = Camera[2].vpx;
                  Camera[2].vrz = CarModel.Object_Coord.coord.t[2]; break;};
      case 3    : {                                   // overhead camera
                  FntPrint("Overhead camera\n"); break;};
      case 4    : {                                   // in car camera
                  FntPrint("In car camera\n"); break;};
      case 5    : {                                   // blimp camera
                  FntPrint("Blimp camera\n");
                  Camera[5].vrx = CarModel.Object_Coord.coord.t[0];
                  Camera[5].vry = CarModel.Object_Coord.coord.t[1];
                  Camera[5].vrz = CarModel.Object_Coord.coord.t[2]; break;};
      case 6    : {                                   // sidecar camera
                  FntPrint("Sidecar camera\n"); break;};
      };
    FntPrint("\nUse all eight buttons to change camera\n");
    FntPrint("Vertical sync value (%d)\n", VerticalSync);
                                                      // display speed of screen drawing
    ProcessUserInput();                               // call function to process joypad
    MoveModelForward(&CarModel, CarModel.speed);      // move man every frame
    RenderWorld();                                    // render the playstation screen
    }
  ResetGraph(3);                                      // clear up before quitting
  };

// ***********************************************************************************
// User defined drawing functions
// ***********************************************************************************

// **** Actual main screen drawing routine (calls all functions)
void RenderWorld()
  {
  int currentBuffer = GsGetActiveBuff();              // determine which buffer is to be used
  GsSetRefView2(&Camera[cameraUsed]);                           // activates the view
  GsSetWorkBase((PACKET*)Packet_Memory[currentBuffer]);
                                                      // define where screen drawing commands
                                                      // are going to be queued
  GsClearOt(0, 0, &OTable_Header[currentBuffer]);     // clear out OTable sorting info
  DrawTrack(&TrackData, &OTable_Header[currentBuffer]);
  DrawObject(&CarModel, &OTable_Header[currentBuffer]);
                                                      // send draw commands to queue
  FntFlush(-1);                                       // force text to be printed immediately
  DrawSync(0);                                        // wait for completion of all output
  VerticalSync = VSync(0);                            // wait for blank and time it
  GsSwapDispBuff();                                   // swap buffer information
  GsSortClear(0, 0, 0,&OTable_Header[currentBuffer]); // send clear screen command to queue
  GsDrawOt(&OTable_Header[currentBuffer]);            // draw commands in queue
  };

// **** draw an object on screen
void DrawObject(Object_Header0 *tObject, GsOT *tOTable)
  {
  MATRIX  tLocalWorld, tLocalScreen;                  // create two MATRIX structures
  GsGetLws(tObject->Object_Handler.coord2, &tLocalWorld, &tLocalScreen);
                                                      // get the local world and screen
                                                      // matrices. Needed for lightsourcing
  GsSetLightMatrix(&tLocalWorld);                     // Set local world matrix for lighting
  GsSetLsMatrix(&tLocalScreen);                       // Set local screen matrix for perspective
  GsSortObject4( &tObject->Object_Handler, tOTable, 4, (u_long*)getScratchAddr(0));
                                                      // Put object in OTable ready for drawing
  };

// **** draw the entire track on screen
void DrawTrack(Track_Header0 *tTrack, GsOT *tOTable)
  {
  MATRIX tLocalWorld, tLocalScreen;                   // create two MATRIX structures
  int tCounter;
  for (tCounter = 0; tCounter < tTrack->numberObjects; tCounter++)
    {                                                 // go through every cell in data array
    GsGetLws(tTrack->Object_Handler[tCounter].coord2, &tLocalWorld, &tLocalScreen);
                                                      // get local world and screen
                                                      // matrices. Needed for lightsourcing
    GsSetLightMatrix(&tLocalWorld);                   // Set local world matrix for lighting
    GsSetLsMatrix(&tLocalScreen);                     // Set local screen matrix for perspective
    GsSortObject4( &tTrack->Object_Handler[tCounter], tOTable,
                    2, (u_long*)getScratchAddr(0));   // Put object in OTable ready for drawing
                                                      // their value of 2 is lower than the
                                                      // car's value of 4, therefore car is
                                                      // drawn after ground which is correct
    };
  };

// ***********************************************************************************
// User defined initialisation functions
// ***********************************************************************************

// **** Load in and setup a .TIM picture file (in this case to be used for a texture)
void LoadTexture(long tMemAddress)
  {
  RECT tRect;                                         // rectangular area to be used for TIM
  GsIMAGE tTim;                                       // TIM image information
  tMemAddress += 4;                                   // advance memory pointer to data
  GsGetTimInfo((u_long *) tMemAddress, &tTim);        // fill tTim with info from TIM
  tRect.x = tTim.px;                                  // tell GPU where graphics data is
  tRect.y = tTim.py;
  tRect.w = tTim.pw;
  tRect.h = tTim.ph;
  LoadImage(&tRect, tTim.pixel);                      // load video memory with TIM data
  if((tTim.pmode>>3) & 0x01)                          // do this if TIM uses a CLUT
    {
    tRect.x = tTim.cx;                                // tell GPU where colour table is
    tRect.y = tTim.cy;
    tRect.w = tTim.cw;
    tRect.h = tTim.ch;
    LoadImage(&tRect, tTim.clut);                     // load video memory with CLUT data
    }
  DrawSync(0);                                        // force LoadImages to complete
  }

// **** Setup up screen
void InitialiseGraphics()
  {
  if (IS_PAL_USED)                                    // set up PAL screen
    {
    SetVideoMode(MODE_PAL);                           // define screen as PAL
    GsInitGraph(PAL_WIDTH, PAL_HEIGHT, GsINTER|GsOFSGPU, 1, 0);
                                                      // define screen resolution, whether
                                                      // interlaced, dithered & colour depth
    GsDefDispBuff(0, 0, 0, PAL_HEIGHT);               // define top left corners of
                                                      // both screen buffers in memory
    }
  else                                                // OR set up NTSC screen
    {
    SetVideoMode(MODE_NTSC);                          // define screen as NTSC
    GsInitGraph(NTSC_WIDTH, NTSC_HEIGHT, GsINTER|GsOFSGPU, 1, 0);
    GsDefDispBuff(0, 0, 0, NTSC_HEIGHT);              // define top left corners of
                                                      // both screen buffers in memory
    };
  GsInit3D();                                         // initialise 3d graphics routine so
                                                      // 3d and font functions can work
  OTable_Header[0].length = OTABLE_LENGTH;            // notify OTable headers of OTable
  OTable_Header[1].length = OTABLE_LENGTH;            // array lengths
  OTable_Header[0].org = OTable_Array[0];             // notify OTable headers of OTable
  OTable_Header[1].org = OTable_Array[1];             // array memory locations
  GsClearOt(0, 0, &OTable_Header[0]);                 // clear out OTable sorting info
  GsClearOt(0, 0, &OTable_Header[1]);                 // ready for use (not strictly necessary
                                                      // as they are cleared before use in
                                                      // RenderWorld()
  };

// **** Initialise a new object - ie, set up its rotation, position and model
void InitModel(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress)
  {
  tObject->rotation.vx = 0;                           // define orientation as zero
  tObject->rotation.vy = 0;
  tObject->rotation.vz = 0;
  tObject->speed = 0;                                 // define velocity of car as zero
  AddModelToHeader(tObject, tX, tY, tZ, tMemAddress); // set up rest of object
  };

// **** Add infomation to object header to specify a 3d model to be associated with it
void AddModelToHeader(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress)
  {
  tMemAddress++;                                      // move pointer past model Id
  GsMapModelingData(tMemAddress);                     // specify that .TMD data exists
                                                      // once past the model Id
  GsInitCoordinate2(WORLD, &tObject->Object_Coord);   // set objects coord system to that
                                                      // of the world (ie, 0, 0, 0)
  tMemAddress += 2;                                   // move pointer to actual 3D data
  GsLinkObject4((u_long)tMemAddress, &tObject->Object_Handler, 0);
                                                      // associate the model with it's handler
  tObject->Object_Handler.coord2 = &tObject->Object_Coord;
                                                      // associate coords of model to
                                                      // it's handler
  tObject->Object_Coord.coord.t[0] = tX;              // set model's initial X translation
  tObject->Object_Coord.coord.t[1] = tY;              //   Y
  tObject->Object_Coord.coord.t[2] = tZ;              //   Z
  tObject->Object_Coord.flg = 0;                      // Set to zero so object is to be updated
  };

// Initialise the Track
void InitTrack()
  {
  long tXpos, tZpos;                                  // temp variables
  char tTrackStatus;                                  // temp variable
  TrackData.numberObjects = 0;                        // define no objects in world to start
  LoadTexture(ROAD_TEX_MEM_ADDR);
  for (tXpos = 0; tXpos<GROUND_MAXX; tXpos++)         // this loop will go through every cell
    {                                                 // in the RawGroundArray array
    for (tZpos = 0; tZpos<GROUND_MAXZ; tZpos++)
      {
      tTrackStatus = RawGroundArray[tXpos][tZpos];    // read cell value in array
      if (tTrackStatus == '1')                        // if a road is designated there
        {
        AddModelToTrack(&TrackData, (tXpos * GROUND_SIZE), 0, (tZpos * GROUND_SIZE),
                                    (u_long *) ROAD_MEM_ADDR);
                                                      // call function to create read object
        };
      };
    };
  };

// **** Add information to track header t specify a 3d model to be associated with it
void AddModelToTrack(Track_Header0 *tTrack, int tX, int tY, int tZ, u_long *tMemAddress)
  {
  long tNumber = tTrack->numberObjects;               // determine which object is being handled
  tTrack->Object_Pointer[tNumber] = (u_long *) tMemAddress;
                                                      // tell model where road TIM is
  tTrack->Object_Pointer[tNumber]++;                  // advance pointer past initial info
  GsMapModelingData(tTrack->Object_Pointer[tNumber]); // specify that TIM data exists
  GsInitCoordinate2(WORLD, &tTrack->Object_Coord[tNumber]);
                                                      // signify object origin = WORLD origin
  tTrack->Object_Pointer[tNumber] += 2;               // advance pointer again! (standard)
  GsLinkObject4((u_long)tTrack->Object_Pointer[tNumber], &tTrack->Object_Handler[tNumber], 0);
                                                      // associate the model with it's handler
  tTrack->Object_Handler[tNumber].attribute = GsDIV1; // signify polygon subdivision occurs
  tTrack->Object_Handler[tNumber].coord2 = &tTrack->Object_Coord[tNumber];
                                                      // associate coords of model to handler
  tTrack->Object_Coord[tNumber].coord.t[0] = tX;      // setup initial position of model
  tTrack->Object_Coord[tNumber].coord.t[1] = tY;
  tTrack->Object_Coord[tNumber].coord.t[2] = tZ;
  tTrack->Object_Coord[tNumber].flg = 0;              // signify it has to be recalculated
  tTrack->numberObjects++;                            // increase counter for next object
  };

// **** Define all lights that will be used in object rendering
void InitAllLights()
  {
  InitLight(&LightSource, 0, 1, 1, 0, 127, 127, 127); // define light
  GsSetAmbient(2047,2047,2047);                       // set ambient light in the scene
  };

// **** Explicitly define a light source (up to three maximum in scene)
void InitLight(GsF_LIGHT *tLight, int tNum, int tXv, int tYv, int tZv, int tR, int tG, int tB)
  {
  tLight->vx = tXv;     tLight->vy = tYv;     tLight->vz = tZv;
                                                      // define vector for light from source
  tLight->r = tR;       tLight->g = tG;       tLight->b = tB;
                                                      // define RGB values for the light
  GsSetFlatLight(tNum, tLight);                       // assign light to an ID number (0-2)
                                                      // and turning light on
  };

// **** Define all information for the viewpoint
void InitView(GsRVIEW2 *tView, int tProjDist, int tRZ,
                  int tFromX, int tFromY, int tFromZ,
                  int tToX, int tToY, int tToZ,
                  GsCOORDINATE2 *tReference)
  {
  GsSetProjection(tProjDist);                         // distance from view projection screen
  tView->vpx = tFromX;  tView->vpy = tFromY;  tView->vpz = tFromZ;
                                                      // set up position to look from
  tView->vrx = tToX;    tView->vry = tToY;    tView->vrz = tToZ;
                                                      // set up position to look to
  tView->rz=-tRZ;                                     // define rotation about view line
  if (tReference != false) tView->super = tReference; // set origin of model
  else tView->super = WORLD;
  };

// ***********************************************************************************
// User defined miscellaneous functions
// ***********************************************************************************

// **** Let Yaroze respond to joypad input
void ProcessUserInput()
  {
  u_long PAD = PadRead();                             // read status of joypad 1
  if (PAD& PADselect)   Running = false;              // select will quit
  if (PAD& PADstart)    {                             // return car to center, and stop motion
                        CarModel.speed = 0;           // set car speed to zero
                        CarModel.rotation.vx = 0;     // straighten car and reset position
                        CarModel.rotation.vy = 0;
                        CarModel.rotation.vz = 0;
                        RotateModel(&CarModel, 0, 0, 0);
                        MoveModelTo(&CarModel, 0, 0, 0);};
  if (PAD& PADup)       {                             // move car away from viewer
                        if (CarModel.speed<100) CarModel.speed++;};
  if (PAD& PADdown)     {                             // move car towards viewer
                        if (CarModel.speed>-100) CarModel.speed--;};
  if (PAD& PADleft)     {                             // rotate car anti-clockwise
                        RotateModel(&CarModel, 0, -16, 0);};
  if (PAD& PADright)    {                             // rotate car clockwise
                        RotateModel(&CarModel, 0, 16, 0);};
  if (PAD& PADsquare)   {SelectCamera(0);             // select static camera
                        cameraCounter = 0;};
  if (PAD& PADtriangle) {SelectCamera(1);             // select tracker camera
                        cameraCounter = 0;};
  if (PAD& PADcircle)   {SelectCamera(2);             // select side tracker camera
                        cameraCounter = 0;};
  if (PAD& PADcross)    {SelectCamera(3);             // select overhead camera
                        cameraCounter = 0;};
  if (PAD& PADL1)       {SelectCamera(4);             // select in car camera
                        cameraCounter = 0;};
  if (PAD& PADL2)       {SelectCamera(5);             // select blimp camera
                        cameraCounter = 0;};
  if (PAD& PADR1)       {SelectCamera(6);             // select sidecar camera
                        cameraCounter = 0;};
  if ((PAD& PADR2) && (cameraCounter == 0))
                        {cameraCounter = 150;         // select automatic camera
                        cameraUsed = (cameraUsed+1) % 7;
                        SelectCamera(cameraUsed);};
  };

// **** Move an object to a coordinate
void MoveModelTo(Object_Header0 *tObject, int tXc, int tYc, int tZc)
  {
  tObject->Object_Coord.coord.t[0] = tXc;             // set the object's x coord
  tObject->Object_Coord.coord.t[1] = tYc;             // set the object's y coord
  tObject->Object_Coord.coord.t[2] = tZc;             // set the object's z coord
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };

// **** Move an object relative to where it is at the moment
void MoveModelBy(Object_Header0 *tObject, int tXv, int tYv, int tZv)
  {
  tObject->Object_Coord.coord.t[0] += tXv;            // add to the object's x coord
  tObject->Object_Coord.coord.t[1] += tYv;            // add to the object's y coord
  tObject->Object_Coord.coord.t[2] += tZv;            // add to the object's z coord
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };

// **** Translate an object according to the direction it is facing
void MoveModelForward(Object_Header0 *tObject, int tSpeed)
  {
  MATRIX tMatrix;                                     // temporary rotation matrix
  SVECTOR tOneUnitForward, tDirectionResult;          // two vectors for rotation
  if(tSpeed != 0)                                     // as long as speed is not zero,
    {                                                 // (would cause divide by zero error)
    tOneUnitForward.vx = 0;                           // to move forward = no x translation
    tOneUnitForward.vy = 0;                           // to move forward = no y translation
    tOneUnitForward.vz = ONE;                         // to move forward = 4096 z translation
    RotMatrix(&tObject->rotation, &tMatrix);          // turn vector into rotation matrix
    ApplyMatrixSV(&tMatrix, &tOneUnitForward, &tDirectionResult);
                                                      // multiply our direction vector with
                                                      // the rotation matrix and give the
                                                      // result as another direction vector
    tObject->Object_Coord.coord.t[0] += tDirectionResult.vx * tSpeed / 4096;
    tObject->Object_Coord.coord.t[1] += tDirectionResult.vy * tSpeed / 4096;
    tObject->Object_Coord.coord.t[2] += tDirectionResult.vz * tSpeed / 4096;
    tObject->Object_Coord.flg = 0;                    // signify object has changed
    };
  };

// **** Rotate an object about it's own center
void RotateModel(Object_Header0 *tObject, int tXr, int tYr, int tZr)
  {
  MATRIX tMatrix;                                     // temporary rotation matrix
  ResetRotation(tObject->Object_Coord.coord.m);       // reset rotation part of objects
                                                      // rotation and translation matrix
  tObject->rotation.vx = (tObject->rotation.vx + tXr) % ONE;
                                                      // set up rotation about the axes
  tObject->rotation.vy = (tObject->rotation.vy + tYr) % ONE;
  tObject->rotation.vz = (tObject->rotation.vz + tZr) % ONE;

  RotMatrix(&tObject->rotation, &tMatrix);            // turn vector into rotation matrix
  MulMatrix0(&tObject->Object_Coord.coord, &tMatrix, &tObject->Object_Coord.coord);
                                                      // multiply object and rotation matrices
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };

// **** Reset a three by three rotation/scaling matrix
void ResetRotation(short tArray[3][3])
  {
  tArray[0][0]=tArray[1][1]=tArray[2][2]=ONE;         // turn scalars to one
  tArray[0][1]=tArray[0][2]=tArray[1][0]=tArray[1][2]=tArray[2][0]=tArray[2][1]=0;
                                                      // turn other cells into zero
  };

// **** set up the camera and the correct lighting
void SelectCamera(long tNumber)
  {
  switch(tNumber)                                     // do routine based on which camera...
    {
    case 0   :  {Fogging.dqa = -5000;                 // set up reasonable fogging
                GsSetFogParam(&Fogging);              // activate new fogging level
                cameraUsed = 0; break;};              // set the current camera
    case 1   :  {Fogging.dqa = -5000;                 // set up reasonable fogging
                GsSetFogParam(&Fogging);              // activate new fogging level
                cameraUsed = 1; break;};              // set the current camera
    case 2   :  {Fogging.dqa = -10000;                // set up small fogging
                GsSetFogParam(&Fogging);              // activate new fogging level
                cameraUsed = 2; break;};              // set the current camera
    case 3   :  {Fogging.dqa = -15000;                // set up very little fogging
                GsSetFogParam(&Fogging);              // activate new fogging level
                cameraUsed = 3; break;};              // set the current camera
    case 4   :  {Fogging.dqa = -5000;                 // set up reasonable fogging
                GsSetFogParam(&Fogging);              // activate new fogging level
                cameraUsed = 4; break;};              // set the current camera
    case 5   :  {Fogging.dqa = -15000;                // set up very little fogging
                GsSetFogParam(&Fogging);              // activate new fogging level
                cameraUsed = 5; break;};              // set the current camera
    case 6   :  {Fogging.dqa = -5000;                 // set up reasonable fogging
                GsSetFogParam(&Fogging);              // activate new fogging level
                cameraUsed = 6; break;};              // set the current camera
    };
  };
