// ***********************************************************************************
// Programs written by R.Swan - rs108@mdx.ac.uk - www.netyaroze-europe.com/~middex2
// Taken from the tutorial by P.Passmore - p.passmore@mdx.ac.uk
//   Tutorial file can be found at the above web site
// Step four - move the car and lightsources through the Yaroze onto the TV/Monitor
// ***********************************************************************************

// ***********************************************************************************
// Preprocessor functions
// ***********************************************************************************

// **** include libraries and files
#include <stdio.h>                                    // load standard libraries
#include <libps.h>                                    // load playstation library
#include "pad.h"                                      // load joypad reading

// **** set up defines
#define true          (1)
#define false         (0)
#define OTABLE_LENGTH (12)                            // define length of OTable
#define MAX_PACKETS   (248000)                        // define size of GPU scratchpad
#define IS_PAL_USED   (true)                          // true = PAL mode, false = NTSC mode
#define PAL_WIDTH     (320)                           // size of PAL screen
#define PAL_HEIGHT    (240)
#define NTSC_WIDTH    (320)                           // size of NTSC screen
#define NTSC_HEIGHT   (256)

#define CAR_MEM_ADDR  (0x80090000)                    // location of car .TMD data

// **** create global variables, structures, headers and arrays
GsOT        OTable_Header[2];                         // Header files holding OTable info
GsOT_TAG    OTable_Array[2][1<<OTABLE_LENGTH];        // Array to hold drawing order info
PACKET      Packet_Memory[2][MAX_PACKETS];            // Memory to hold GPU packet data

u_long      PADstatus=0;                              // variable holds state of joypad 1

typedef struct                                        // create structure called
  {                                                   // Object_Header0
  GsDOBJ2         Object_Handler;                     //   contains model info
  GsCOORDINATE2   Object_Coord;                       //   contains object location and
                                                      //   orientation info
  } Object_Header0;
Object_Header0    CarModel;                           // create instance of structure for car
GsF_LIGHT   LightSource[2];                           // Array of two light sources
GsRVIEW2    ViewPoint;                                // structure holding viewpoint info

// **** function prototyping
void AddModelToHeader(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress);
                                                      // join a 3d model with it's info
void InitAllLights();                                 // define all lights to be used
void InitLight(GsF_LIGHT *tLight, int tNum, int tXv, int tYv, int tZv, int tR, int tG, int tB);
                                                      // set information for a lightsource
void InitView(GsRVIEW2 *tView, int tProjDist, int tRZ,// define details of scene viewpoint
                  int tFromX, int tFromY, int tFromZ,
                  int tToX, int tToY, int tToZ);
void InitialiseGraphics();                            // define graphics initialisation
void RenderWorld();                                   // define how to draw the screen
void DrawObject(Object_Header0 *tObject, GsOT *tOTable);
                                                      // send a draw object command to queue
void MoveModelTo(Object_Header0 *tObject, int tXc, int tYc, int tZc);
                                                      // move object to (tXc, tXy, tZc)
void MoveModelBy(Object_Header0 *tObject, int tXv, int tYv, int tZv);
                                                      // translate an object by a vector
void RotateModel(Object_Header0 *tObject, int tXr, int tYr, int tZr);
                                                      // rotate an object (4096 = 1 dergree)

// ***********************************************************************************
// Main function
// ***********************************************************************************

void main()
  {
  int Running = true;                                 // define program to be running
  FntLoad(960, 256);                                  // load font into area of frame
                                                      // specified by 960, 256
  FntOpen(-160, -120, 320, 100, 0, 512);              // specify where on screen text goes
                                                      // (0, 0) is center of screen
  PadInit();                                          // initialise joypad reading
  InitialiseGraphics();                               // initialise graphics
  InitView(&ViewPoint, 250, 0, 0,-500,-1000, 0,0,0 ); // set position & orientation of viewpoint
  InitAllLights();                                    // call function that creates two lights
  AddModelToHeader(&CarModel,  0, -200, 0, (long *)CAR_MEM_ADDR);
                                                      // attach model location to object geader
                                                      // car's position set to (0, -200, 0);

  while(Running)                                      // do loop while program is running
    {
    FntPrint("JOYPAD to move\nL1/L2 to rotate\nSTART to recenter\nSELECT to exit\n\n");
                                                      // send print command to queue
// **** Interaction with the joypad -
//      SELECT = quit, START = Move the car back to the center of screen,
//      JOYPAD moving = move object, L1 and R1 = rotate model
    PADstatus = PadRead();                            // read status of joypad 1
    if (PADstatus&PADselect)    Running = false;
    if (PADstatus&PADstart)     MoveModelTo(&CarModel, 0, -200, 0);
    if (PADstatus&PADup)        {MoveModelBy(&CarModel, 0, 0, 5); FntPrint("Move up\n");};
    if (PADstatus&PADdown)      {MoveModelBy(&CarModel, 0, 0, -5); FntPrint("Move down\n");};
    if (PADstatus&PADleft)      {MoveModelBy(&CarModel, -5, 0, 0); FntPrint("Move left\n");};
    if (PADstatus&PADright)     {MoveModelBy(&CarModel, 5, 0, 0);  FntPrint("Move right\n");};
    if (PADstatus&PADL1)        {RotateModel(&CarModel, 0, -16, 0); FntPrint("Turn left\n");};
    if (PADstatus&PADR1)        {RotateModel(&CarModel, 0, 16, 0); FntPrint("Turn right\n");};

    RenderWorld();                                    // render the playstation screen
    }
  ResetGraph(3);                                      // clear up before quitting
  };

// ***********************************************************************************
// User defined drawing functions
// ***********************************************************************************

// **** Actual main screen drawing routine (calls all functions)
void RenderWorld()
  {
  int currentBuffer = GsGetActiveBuff();              // determine which buffer is to be used
  GsSetWorkBase((PACKET*)Packet_Memory[currentBuffer]);
                                                      // define where screen drawing commands
                                                      // are going to be queued
  GsClearOt(0, 0, &OTable_Header[currentBuffer]);     // clear out OTable sorting info
  DrawObject(&CarModel, &OTable_Header[currentBuffer]);
                                                      // send draw car command to queue
  FntFlush(-1);                                       // force text to be printed immediately
  DrawSync(0);                                        // wait for completion of all output
  VSync(0);                                           // wait for vertical blanking
  GsSwapDispBuff();                                   // swap buffer information
  GsSortClear(0, 0, 0,&OTable_Header[currentBuffer]); // send clear screen command to queue
  GsDrawOt(&OTable_Header[currentBuffer]);            // draw commands in queue
  };

// **** draw an object on screen
void DrawObject(Object_Header0 *tObject, GsOT *tOTable)
  {
  MATRIX  tLocalWorld, tLocalScreen;                  // create two MATRIX structures
  GsGetLws(tObject->Object_Handler.coord2, &tLocalWorld, &tLocalScreen);
                                                      // get the local world and screen
                                                      // matrices. Needed for lightsourcing
  GsSetLightMatrix(&tLocalWorld);                     // Set local world matrix for lighting
  GsSetLsMatrix(&tLocalScreen);                       // Set local screen matrix for perspective
  GsSortObject4( &tObject->Object_Handler, tOTable, 4, (u_long*)getScratchAddr(0));
                                                      // Put object in OTable ready for drawing
  };

// ***********************************************************************************
// User defined initialisation functions
// ***********************************************************************************

// **** Setup up screen
void InitialiseGraphics()
  {
  if (IS_PAL_USED)                                    // set up PAL screen
    {
    SetVideoMode(MODE_PAL);                           // define screen as PAL
    GsInitGraph(PAL_WIDTH, PAL_HEIGHT, GsINTER|GsOFSGPU, 1, 0);
                                                      // define screen resolution, whether
                                                      // interlaced, dithered & colour depth
    GsDefDispBuff(0, 0, 0, PAL_HEIGHT);               // define top left corners of
                                                      // both screen buffers in memory
    }
  else                                                // OR set up NTSC screen
    {
    SetVideoMode(MODE_NTSC);                          // define screen as NTSC
    GsInitGraph(NTSC_WIDTH, NTSC_HEIGHT, GsINTER|GsOFSGPU, 1, 0);
    GsDefDispBuff(0, 0, 0, NTSC_HEIGHT);              // define top left corners of
                                                      // both screen buffers in memory
    };
  GsInit3D();                                         // initialise 3d graphics routine so
                                                      // 3d and font functions can work
  OTable_Header[0].length = OTABLE_LENGTH;            // notify OTable headers of OTable
  OTable_Header[1].length = OTABLE_LENGTH;            // array lengths
  OTable_Header[0].org = OTable_Array[0];             // notify OTable headers of OTable
  OTable_Header[1].org = OTable_Array[1];             // array memory locations
  GsClearOt(0, 0, &OTable_Header[0]);                 // clear out OTable sorting info
  GsClearOt(0, 0, &OTable_Header[1]);                 // ready for use (not strictly necessary
                                                      // as they are cleared before use in
                                                      // RenderWorld()
  };

// **** Add infomation to object header to specify a 3d model to be associated with it
void AddModelToHeader(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress)
  {
  tMemAddress++;                                      // move pointer past model Id
  GsMapModelingData(tMemAddress);                     // specify that .TMD data exists
                                                      // once past the model Id
  GsInitCoordinate2(WORLD, &tObject->Object_Coord);   // set objects coord system to that
                                                      // of the world (ie, 0, 0, 0)
  tMemAddress += 2;                                   // move pointer to actual 3D data
  GsLinkObject4((u_long)tMemAddress, &tObject->Object_Handler, 0);
                                                      // associate the model with it's handler
  tObject->Object_Handler.coord2 = &tObject->Object_Coord;
                                                      // associate coords of model to
                                                      // it's handler
  tObject->Object_Coord.coord.t[0] = tX;              // set model's initial X translation
  tObject->Object_Coord.coord.t[1] = tY;              //   Y
  tObject->Object_Coord.coord.t[2] = tZ;              //   Z
  tObject->Object_Coord.flg = 0;                      // Set to zero so object is to be updated
  };

// **** Define all lights that will be used in object rendering
void InitAllLights()
  {
  InitLight(&LightSource[0], 0, -1, -1, -1, 255, 255, 255);
                                                      // define each light, one at a time
  InitLight(&LightSource[1], 1, 1, 1, 1, 255, 255, 255);
  GsSetAmbient(0,0,0);                                // set ambient light in the scene
  GsSetLightMode(0);                                  // set scene to lighting without fog
  };

// **** Explicitly define a light source (up to three maximum in scene)
void InitLight(GsF_LIGHT *tLight, int tNum, int tXv, int tYv, int tZv, int tR, int tG, int tB)
  {
  tLight->vx = tXv;     tLight->vy = tYv;     tLight->vz = tZv;
                                                      // define vector for light from source
  tLight->r = tR;       tLight->g = tG;       tLight->b = tB;
                                                      // define RGB values for the light
  GsSetFlatLight(tNum, tLight);                       // assign light to an ID number (0-2)
                                                      // and turning light on
  };

// **** Define all information for the viewpoint
void InitView(GsRVIEW2 *tView, int tProjDist, int tRZ,
                  int tFromX, int tFromY, int tFromZ,
                  int tToX, int tToY, int tToZ)
  {
  GsSetProjection(tProjDist);                         // distance from view projection screen
  tView->vpx = tFromX;  tView->vpy = tFromY;  tView->vpz = tFromZ;
                                                      // set up position to look from
  tView->vrx = tToX;    tView->vry = tToY;    tView->vrz = tToZ;
                                                      // set up position to look to
  tView->rz=-tRZ;                                     // define rotation about view line
  tView->super = WORLD;                               // set origin of model to that of world
  GsSetRefView2(tView);                               // activates the view
  };

// ***********************************************************************************
// User defined miscellaneous functions
// ***********************************************************************************

// **** Move an object to a coordinate
void MoveModelTo(Object_Header0 *tObject, int tXc, int tYc, int tZc)
  {
  tObject->Object_Coord.coord.t[0] = tXc;             // set the object's x coord
  tObject->Object_Coord.coord.t[1] = tYc;             // set the object's y coord
  tObject->Object_Coord.coord.t[2] = tZc;             // set the object's z coord
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };

// **** Move an object relative to where it is at the moment
void MoveModelBy(Object_Header0 *tObject, int tXv, int tYv, int tZv)
  {
  tObject->Object_Coord.coord.t[0] += tXv;            // add to the object's x coord
  tObject->Object_Coord.coord.t[1] += tYv;            // add to the object's y coord
  tObject->Object_Coord.coord.t[2] += tZv;            // add to the object's z coord
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };

// **** Rotate an object about it's own center
void RotateModel(Object_Header0 *tObject, int tXr, int tYr, int tZr)
  {
  MATRIX tMatrix;
  SVECTOR tRotation;

  tRotation.vx = tXr;
  tRotation.vy = tYr;
  tRotation.vz = tZr;

  RotMatrix(&tRotation, &tMatrix);
  MulMatrix0(&tObject->Object_Coord.coord, &tMatrix, &tObject->Object_Coord.coord);
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };
