// ***********************************************************************************
// Programs written by R.Swan - rs108@mdx.ac.uk - www.netyaroze-europe.com/~middex2
// Taken from the tutorial by P.Passmore - p.passmore@mdx.ac.uk
//   Tutorial file can be found at the above web site
// Step eight - adding textures to the car
// ***********************************************************************************

// ***********************************************************************************
// Preprocessor functions
// ***********************************************************************************

// **** include libraries and files
#include <stdio.h>                                    // load standard libraries
#include <libps.h>                                    // load playstation library
#include "pad.h"                                      // load joypad reading

// **** set up defines
#define true          (1)
#define false         (0)
#define OTABLE_LENGTH (12)                            // define length of OTable
#define MAX_PACKETS   (248000)                        // define size of GPU scratchpad
#define IS_PAL_USED   (true)                          // true = PAL mode, false = NTSC mode
#define PAL_WIDTH     (320)                           // size of PAL screen
#define PAL_HEIGHT    (240)
#define NTSC_WIDTH    (320)                           // size of NTSC screen
#define NTSC_HEIGHT   (256)

#define CAR_MEM_ADDR  (0x80090000)                    // location of car .TMD data
#define TEX_MEM_ADDR  (0x800A0000)                    // location of texture .TIM data

// **** create global variables, structures, headers and arrays (mostly system stuff)
int         Running = true;                           // define program to be running
u_long      VerticalSync = 0;                         // set up speed counter for processor

GsOT        OTable_Header[2];                         // Header files holding OTable info
GsOT_TAG    OTable_Array[2][1<<OTABLE_LENGTH];        // Array to hold drawing order info
PACKET      Packet_Memory[2][MAX_PACKETS];            // Memory to hold GPU packet data

typedef struct                                        // create structure called
  {                                                   // Object_Header0
  SVECTOR         rotation;                           //   vector array to hold rotation
  long            speed;                              //   integer holding velocity of car
  GsDOBJ2         Object_Handler;                     //   contains model info
  GsCOORDINATE2   Object_Coord;                       //   contains object location and
                                                      //   orientation info
  } Object_Header0;
Object_Header0    CarModel;                           // create instance of structure for car
GsF_LIGHT   LightSource[2];                           // Array of two light sources
GsRVIEW2    *ViewPoint;                               // pointer to camera that is being used
GsRVIEW2    StaticView;                               // structure holding static camera info
GsRVIEW2    TrackerView;                              // structure holding tracking camera info

// **** function prototyping
void ProcessUserInput();                              // react to joypad use
void LoadTexture(long tMemAddress);                   // load a texture into video memory
void InitModel(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress);
                                                      // general object initialisation function
void AddModelToHeader(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress);
                                                      // join a 3d model with it's info
void InitAllLights();                                 // define all lights to be used
void InitLight(GsF_LIGHT *tLight, int tNum, int tXv, int tYv, int tZv, int tR, int tG, int tB);
                                                      // set information for a lightsource
void InitView(GsRVIEW2 *tView, int tProjDist, int tRZ,// define details of scene viewpoint
                  int tFromX, int tFromY, int tFromZ,
                  int tToX, int tToY, int tToZ,
                  GsCOORDINATE2 *tReference);
void InitialiseGraphics();                            // define graphics initialisation
void RenderWorld();                                   // define how to draw the screen
void DrawObject(Object_Header0 *tObject, GsOT *tOTable);
                                                      // send a draw object command to queue
void MoveModelTo(Object_Header0 *tObject, int tXc, int tYc, int tZc);
                                                      // move object to (tXc, tXy, tZc)
void MoveModelBy(Object_Header0 *tObject, int tXv, int tYv, int tZv);
                                                      // translate an object by a vector
void MoveModelForward(Object_Header0 *tObject, int tSpeed);
                                                      // move object in direction it faces
void RotateModel(Object_Header0 *tObject, int tXr, int tYr, int tZr);
                                                      // rotate an object (4096 = 360 degrees)
void ResetRotation(short tArray[3][3]);               // reset rotation part of a matrix

// ***********************************************************************************
// Main function
// ***********************************************************************************

void main()
  {
  FntLoad(960, 256);                                  // load font into area of frame
                                                      // specified by 960, 256
  FntOpen(-160, -120, 320, 100, 0, 512);              // specify where on screen text goes
                                                      // (0, 0) is center of screen
  PadInit();                                          // initialise joypad reading
  InitialiseGraphics();                               // initialise graphics
  InitView(&StaticView, 250, 0, 0,-500,-1000, 0,0,0 ,false);
                                                      // set static camera
  InitView(&TrackerView, 250, 0, 0,-500,-1000, 0,0,0 ,&CarModel.Object_Coord);
                                                      // set tracker camera
  ViewPoint = &StaticView;                            // use static camera to begin with
  InitAllLights();                                    // call function that creates two lights
  InitModel(&CarModel,  0, -200, 0, (u_long *)CAR_MEM_ADDR);
                                                      // prepare object for use
                                                      // car's position set to (0, -200, 0);
  LoadTexture(TEX_MEM_ADDR);                          // load the number plate texture

  while(Running)                                      // do loop while program is running
    {
    FntPrint("up/down to move\nleft/right rotate\nTriangle for tracker view\n");
    FntPrint("Square for static view\nSTART to recenter\nSELECT to exit\n\n");
                                                      // send print command to queue
    FntPrint("Vertical sync value (%d)\n", VerticalSync);
                                                      // display speed of screen drawing
    FntPrint("Car Coordinates (%d, %d, %d)\n"         // display car x,y,z coords on screen
                            ,CarModel.Object_Coord.coord.t[0]
                            ,CarModel.Object_Coord.coord.t[1]
                            ,CarModel.Object_Coord.coord.t[2]);
    FntPrint("Car speed (%d)\n\n", CarModel.speed);
                                                      // display car speed
    ProcessUserInput();                               // call function to process joypad
    MoveModelForward(&CarModel, CarModel.speed);      // move car every frame
    RenderWorld();                                    // render the playstation screen
    }
  ResetGraph(3);                                      // clear up before quitting
  };

// ***********************************************************************************
// User defined drawing functions
// ***********************************************************************************

// **** Actual main screen drawing routine (calls all functions)
void RenderWorld()
  {
  int currentBuffer = GsGetActiveBuff();              // determine which buffer is to be used
  GsSetRefView2(ViewPoint);                           // activates the view
  GsSetWorkBase((PACKET*)Packet_Memory[currentBuffer]);
                                                      // define where screen drawing commands
                                                      // are going to be queued
  GsClearOt(0, 0, &OTable_Header[currentBuffer]);     // clear out OTable sorting info
  DrawObject(&CarModel, &OTable_Header[currentBuffer]);
                                                      // send draw car command to queue
  FntFlush(-1);                                       // force text to be printed immediately
  DrawSync(0);                                        // wait for completion of all output
  VerticalSync = VSync(0);                            // wait for blank and time it
  GsSwapDispBuff();                                   // swap buffer information
  GsSortClear(0, 0, 0,&OTable_Header[currentBuffer]); // send clear screen command to queue
  GsDrawOt(&OTable_Header[currentBuffer]);            // draw commands in queue
  };

// **** draw an object on screen
void DrawObject(Object_Header0 *tObject, GsOT *tOTable)
  {
  MATRIX  tLocalWorld, tLocalScreen;                  // create two MATRIX structures
  GsGetLws(tObject->Object_Handler.coord2, &tLocalWorld, &tLocalScreen);
                                                      // get the local world and screen
                                                      // matrices. Needed for lightsourcing
  GsSetLightMatrix(&tLocalWorld);                     // Set local world matrix for lighting
  GsSetLsMatrix(&tLocalScreen);                       // Set local screen matrix for perspective
  GsSortObject4( &tObject->Object_Handler, tOTable, 4, (u_long*)getScratchAddr(0));
                                                      // Put object in OTable ready for drawing
  };

// ***********************************************************************************
// User defined initialisation functions
// ***********************************************************************************

// **** Load in and setup a .TIM picture file (in this case to be used for a texture)
void LoadTexture(long tMemAddress)
  {
  RECT tRect;                                         // rectangular area to be used for TIM
  GsIMAGE tTim;                                       // TIM image information
  tMemAddress += 4;                                    // advance memory pointer to data
  GsGetTimInfo((u_long *) tMemAddress, &tTim);        // fill tTim with info from TIM
  tRect.x = tTim.px;                                  // tell GPU where graphics data is
  tRect.y = tTim.py;
  tRect.w = tTim.pw;
  tRect.h = tTim.ph;
  LoadImage(&tRect, tTim.pixel);                      // load video memory with TIM data
  if((tTim.pmode>>3) & 0x01)                          // do this if TIM uses a CLUT
    {
    tRect.x = tTim.cx;                                // tell GPU where colour table is
    tRect.y = tTim.cy;
    tRect.w = tTim.cw;
    tRect.h = tTim.ch;
    LoadImage(&tRect, tTim.clut);                     // load video memory with CLUT data
    }
  DrawSync(0);                                        // force LoadImages to complete
  }

// **** Setup up screen
void InitialiseGraphics()
  {
  if (IS_PAL_USED)                                    // set up PAL screen
    {
    SetVideoMode(MODE_PAL);                           // define screen as PAL
    GsInitGraph(PAL_WIDTH, PAL_HEIGHT, GsINTER|GsOFSGPU, 1, 0);
                                                      // define screen resolution, whether
                                                      // interlaced, dithered & colour depth
    GsDefDispBuff(0, 0, 0, PAL_HEIGHT);               // define top left corners of
                                                      // both screen buffers in memory
    }
  else                                                // OR set up NTSC screen
    {
    SetVideoMode(MODE_NTSC);                          // define screen as NTSC
    GsInitGraph(NTSC_WIDTH, NTSC_HEIGHT, GsINTER|GsOFSGPU, 1, 0);
    GsDefDispBuff(0, 0, 0, NTSC_HEIGHT);              // define top left corners of
                                                      // both screen buffers in memory
    };
  GsInit3D();                                         // initialise 3d graphics routine so
                                                      // 3d and font functions can work
  OTable_Header[0].length = OTABLE_LENGTH;            // notify OTable headers of OTable
  OTable_Header[1].length = OTABLE_LENGTH;            // array lengths
  OTable_Header[0].org = OTable_Array[0];             // notify OTable headers of OTable
  OTable_Header[1].org = OTable_Array[1];             // array memory locations
  GsClearOt(0, 0, &OTable_Header[0]);                 // clear out OTable sorting info
  GsClearOt(0, 0, &OTable_Header[1]);                 // ready for use (not strictly necessary
                                                      // as they are cleared before use in
                                                      // RenderWorld()
  };

// **** Initialise a new object - ie, set up its rotation, position and model
void InitModel(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress)
  {
  tObject->rotation.vx = 0;                           // define orientation as zero
  tObject->rotation.vy = 0;
  tObject->rotation.vz = 0;
  tObject->speed = 0;                                 // define velocity of car as zero
  AddModelToHeader(tObject, tX, tY, tZ, tMemAddress); // set up rest of object
  };

// **** Add infomation to object header to specify a 3d model to be associated with it
void AddModelToHeader(Object_Header0 *tObject, int tX, int tY, int tZ, u_long *tMemAddress)
  {
  tMemAddress++;                                      // move pointer past model Id
  GsMapModelingData(tMemAddress);                     // specify that .TMD data exists
                                                      // once past the model Id
  GsInitCoordinate2(WORLD, &tObject->Object_Coord);   // set objects coord system to that
                                                      // of the world (ie, 0, 0, 0)
  tMemAddress += 2;                                   // move pointer to actual 3D data
  GsLinkObject4((u_long)tMemAddress, &tObject->Object_Handler, 0);
                                                      // associate the model with it's handler
  tObject->Object_Handler.coord2 = &tObject->Object_Coord;
                                                      // associate coords of model to
                                                      // it's handler
  tObject->Object_Coord.coord.t[0] = tX;              // set model's initial X translation
  tObject->Object_Coord.coord.t[1] = tY;              //   Y
  tObject->Object_Coord.coord.t[2] = tZ;              //   Z
  tObject->Object_Coord.flg = 0;                      // Set to zero so object is to be updated
  };

// **** Define all lights that will be used in object rendering
void InitAllLights()
  {
  InitLight(&LightSource[0], 0, -1, -1, -1, 255, 255, 255);
                                                      // define each light, one at a time
  InitLight(&LightSource[1], 1, 1, 1, 1, 255, 255, 255);
  GsSetAmbient(0,0,0);                                // set ambient light in the scene
  GsSetLightMode(0);                                  // set scene to lighting without fog
  };

// **** Explicitly define a light source (up to three maximum in scene)
void InitLight(GsF_LIGHT *tLight, int tNum, int tXv, int tYv, int tZv, int tR, int tG, int tB)
  {
  tLight->vx = tXv;     tLight->vy = tYv;     tLight->vz = tZv;
                                                      // define vector for light from source
  tLight->r = tR;       tLight->g = tG;       tLight->b = tB;
                                                      // define RGB values for the light
  GsSetFlatLight(tNum, tLight);                       // assign light to an ID number (0-2)
                                                      // and turning light on
  };

// **** Define all information for the viewpoint
void InitView(GsRVIEW2 *tView, int tProjDist, int tRZ,
                  int tFromX, int tFromY, int tFromZ,
                  int tToX, int tToY, int tToZ,
                  GsCOORDINATE2 *tReference)
  {
  GsSetProjection(tProjDist);                         // distance from view projection screen
  tView->vpx = tFromX;  tView->vpy = tFromY;  tView->vpz = tFromZ;
                                                      // set up position to look from
  tView->vrx = tToX;    tView->vry = tToY;    tView->vrz = tToZ;
                                                      // set up position to look to
  tView->rz=-tRZ;                                     // define rotation about view line
  if (tReference != false) tView->super = tReference; // set origin of model
  else tView->super = WORLD;
  };

// ***********************************************************************************
// User defined miscellaneous functions
// ***********************************************************************************

// **** Let Yaroze respond to joypad input
void ProcessUserInput()
  {
  u_long PAD = PadRead();                             // read status of joypad 1
  if (PAD& PADselect)   Running = false;              // select will quit
  if (PAD& PADstart)    {                             // return car to center, and stop motion
                        CarModel.speed = 0;           // set car speed to zero
                        CarModel.rotation.vx = 0;     // these three lines straighten car
                        CarModel.rotation.vy = 0;
                        CarModel.rotation.vz = 0;
                        RotateModel(&CarModel, 0, 0, 0);
                                                      // forces rotation to be recalculated
                        MoveModelTo(&CarModel, 0, -200, 0);};
  if (PAD& PADup)       {                             // move car away from viewer
                        if (CarModel.speed<20) CarModel.speed++;
                        FntPrint("Move forwards\n");};
  if (PAD& PADdown)     {                             // move car towards viewer
                        if (CarModel.speed>-20) CarModel.speed--;
                        FntPrint("Move backwards\n");};
  if (PAD& PADleft)     {                             // rotate car anti-clockwise
                        RotateModel(&CarModel, 0, -16, 0);
                        FntPrint("Turn left\n");};
  if (PAD& PADright)    {                             // rotate car clockwise
                        RotateModel(&CarModel, 0, 16, 0);
                        FntPrint("Turn right\n");};
  if (PAD& PADtriangle) {                             // select tracker camera
                        ViewPoint = &TrackerView;};
  if (PAD& PADsquare)   {                             // select static camera
                        ViewPoint = &StaticView;};
  };

// **** Move an object to a coordinate
void MoveModelTo(Object_Header0 *tObject, int tXc, int tYc, int tZc)
  {
  tObject->Object_Coord.coord.t[0] = tXc;             // set the object's x coord
  tObject->Object_Coord.coord.t[1] = tYc;             // set the object's y coord
  tObject->Object_Coord.coord.t[2] = tZc;             // set the object's z coord
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };

// **** Move an object relative to where it is at the moment
void MoveModelBy(Object_Header0 *tObject, int tXv, int tYv, int tZv)
  {
  tObject->Object_Coord.coord.t[0] += tXv;            // add to the object's x coord
  tObject->Object_Coord.coord.t[1] += tYv;            // add to the object's y coord
  tObject->Object_Coord.coord.t[2] += tZv;            // add to the object's z coord
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };

// **** Translate an object according to the direction it is facing
void MoveModelForward(Object_Header0 *tObject, int tSpeed)
  {
  MATRIX tMatrix;                                     // temporary rotation matrix
  SVECTOR tOneUnitForward, tDirectionResult;          // two vectors for rotation
  if(tSpeed != 0)                                     // as long as speed is not zero,
    {                                                 // (would cause divide by zero error)
    tOneUnitForward.vx = 0;                           // to move forward = no x translation
    tOneUnitForward.vy = 0;                           // to move forward = no y translation
    tOneUnitForward.vz = ONE;                         // to move forward = 4096 z translation
    RotMatrix(&tObject->rotation, &tMatrix);           // turn vector into rotation matrix
    ApplyMatrixSV(&tMatrix, &tOneUnitForward, &tDirectionResult);
                                                      // multiply our direction vector with
                                                      // the rotation matrix and give the
                                                      // result as another direction vector
    tObject->Object_Coord.coord.t[0] += tDirectionResult.vx * tSpeed / 4096;
    tObject->Object_Coord.coord.t[1] += tDirectionResult.vy * tSpeed / 4096;
    tObject->Object_Coord.coord.t[2] += tDirectionResult.vz * tSpeed / 4096;
    tObject->Object_Coord.flg = 0;
    };
  };

// **** Rotate an object about it's own center
void RotateModel(Object_Header0 *tObject, int tXr, int tYr, int tZr)
  {
  MATRIX tMatrix;                                     // temporary rotation matrix
  ResetRotation(tObject->Object_Coord.coord.m);       // reset rotation part of objects
                                                      // rotation and translation matrix
  tObject->rotation.vx = (tObject->rotation.vx + tXr) % ONE;
                                                      // set up rotation about the axes
  tObject->rotation.vy = (tObject->rotation.vy + tYr) % ONE;
  tObject->rotation.vz = (tObject->rotation.vz + tZr) % ONE;

  RotMatrix(&tObject->rotation, &tMatrix);            // turn vector into rotation matrix
  MulMatrix0(&tObject->Object_Coord.coord, &tMatrix, &tObject->Object_Coord.coord);
                                                      // multiply object and rotation matrices
  tObject->Object_Coord.flg = 0;                      // signify object has changed
  };

// **** Reset a three by three rotation/scaling matrix
void ResetRotation(short tArray[3][3])
  {
  tArray[0][0]=tArray[1][1]=tArray[2][2]=ONE;         // turn scalars to one
  tArray[0][1]=tArray[0][2]=tArray[1][0]=tArray[1][2]=tArray[2][0]=tArray[2][1]=0;
                                                      // turn other cells into zero
  };
