// Filename		: TMDVIEW.C
// Coded by		: Scott Evans
// Created/Modified	: 13/11/97
// Description		: Basic TMD object viewer

#include "tmdview.h"

// Global variables

#ifdef YAROZE

volatile unsigned char *pad0,*pad1;

#endif

GsOT world_ot[2];
GsOT_TAG tags[2][1<<OTSIZE];
PACKET packet_buffer[2][PBSIZE],*pb=(PACKET *)packet_buffer;
DBUFFER db[2],*dbp=db;

// TMD Object 

TMD_OBJECT *object;

// The viewpoint

GsRVIEW2 *view;

// The lights

GsF_LIGHT *lights;

// Lights colours

CVECTOR light_colours[3]={0x90,0x90,0x90,0,
					 0x30,0x30,0x30,0,
					 0x10,0x10,0x10,0};

// Lights angles

SVECTOR light_angles[3]={ONE,0,0,0,
					0,ONE,0,0,
					0,0,ONE,0};

// Fog data

GsFOGPARAM fog_info={0,0,0x0,0x0,0x0};

// Pointer to packet buffer

extern PACKET *GsOUT_PACKET_P;

main(void)
{
	unsigned long pad,hblanks;
	MATRIX m,m1;
	char auto_rotate=0,flag=1;

	// Set up system

	InitSystem(dbp,world_ot,(GsOT_TAG *)tags);
	InitFont(SCRNW,0);
	
	// Set up object
		
	object=InitTMD_OBJECT(TMDADDR,0);
	object->angle.vx=1000;
	object->angle.vy=1040;

	// Set up view

	view=InitRefView(0,0,-1000,0,0,0,0);
	
	// Set lights

	InitLights(light_angles,light_colours);

	// Set light mode (fog+lighting) and background light

	GsSetLightMode(1);
	GsSetAmbient(ONE/4,ONE/4,ONE/4);

	// Set up fog effect

	fog_info.dqa=-4000*ONE/64/SCRNZ;
	fog_info.dqb=5/4*ONE*ONE;

	GsSetFogParam(&fog_info);

	while(1)
	{
		// Swap the double buffers

		dbp=(dbp==db ? db+1 : db);
		
		// Set the packet buffer pointer
		
		GsOUT_PACKET_P=dbp->pb;

		// Clear ordering table

		GsClearOt(0,0,dbp->ot);

		// Set viewpoint

		GsSetRefView2(view);

		// Sort object into ordering table with light source calculation

		GsGetLs(&object->co_system,&m);
		GsSetLsMatrix(&m);
		GsSetLightMatrix(&m);
		
		GsSortObject4(&object->handler,dbp->ot,4,getScratchAddr(0));

		// Wait for drawing to finish

		hblanks=VSync(1);
		VSync(0);
		DrawSync(0);

		GsSwapDispBuff();
		GsSortClear(0,0,0,dbp->ot);

		GsDrawOt(dbp->ot);

		// Read the controllers

		pad=PadRead(0);

		if(pad&PADstart && flag)
		{
			flag=0;

			auto_rotate=++auto_rotate&1;
		}
		
		if(!(pad&PADstart))
			flag=1;

		if(auto_rotate)
		{
			object->angle.vx+=40;
			object->angle.vy+=20;
			object->angle.vz+=10;
		}

		ControlObject(object,pad);
		ControlView(view,pad>>16);		   

		#ifdef YAROZE
		if(pad&PADselect)
			break;
		#endif

		hblanks+=VSync(1);
		FntPrint("Hblanks=%d\n",hblanks);
		FntFlush(-1);
	}

	ResetGraph(3);
	return(0);
}


// Function 		: InitSytem()
// Coded by		: Scott Evans
// Created/Modified	: 13/11/97

// Description		: Initialise the system

// Parameters		: db - pointer to double buffer
//				  ot - pointer to ordering table
//				  tags - pointer to ordering table tags

// Returns		: N/A 

// Notes			: N/A 


void InitSystem(DBUFFER *db,GsOT *ot,GsOT_TAG *tags)
{
	RECT r={0,0,1023,511};

	// Reset graphics system

	ResetGraph(0);
   	
	// Clear VRAM

   	ClearImage(&r,0x0,0x0,0x0);

	// Initialise controllers

	PadInit(0);

	#ifndef YAROZE
	StartPAD(0);
	#endif

	// Initialise ordering table

	ot->length=(ot+1)->length=OTSIZE;
	ot->org=tags;
	(ot+1)->org=tags+(1<<OTSIZE);

	// Set up ordering table and packet buffer

	db->ot=ot;
	(db+1)->ot=ot+1;

	db->pb=pb;
	(db+1)->pb=pb+PBSIZE;

	// Initialise graphics library

	GsInitGraph(SCRNW,SCRNH,GsOFSGPU,0,0);
	GsDefDispBuff(0,0,0,SCRNH);
	GsInit3D();
	
	// Set distance to screen

	GsSetProjection(SCRNZ);
}


// Function 		: InitTMD_OBJECT()
// Coded by		: Scott Evans
// Created/Modified	: 13/11/97

// Description		: Initialise a TMD_OBJECT

// Parameters		: tmd_addr - address of TMD file in memory
//				  n - number of model in TMD file

// Returns		: A pointer to a TMD object

// Notes			: MAX_TMD_OBJECTS sets the maximum number of objects 
//				  that can be used at once, objects are "recycled"


#define MAX_TMD_OBJECTS	20

TMD_OBJECT *InitTMD_OBJECT(unsigned long tmd_addr,int n)
{
	static TMD_OBJECT objects[MAX_TMD_OBJECTS];
	static int object_no=0;
	TMD_OBJECT *objectp=&objects[object_no=++object_no%MAX_TMD_OBJECTS];
	MATRIX m;
	SVECTOR angle={0,0,0};
	VECTOR pos={0,0,0};

	// Change offsets to real addresses

	tmd_addr+=4;
	GsMapModelingData((unsigned long *)tmd_addr);
	
	// Link object 'n' to GsDOBJ2 handler

	tmd_addr+=8;
	GsLinkObject4(tmd_addr,&objectp->handler,n);

	// Set objects angle and position

	RotMatrix(&angle,&m);
	TransMatrix(&m,&pos);

	objectp->co_system.coord=m;
	objectp->co_system.flg=0;
	objectp->co_system.super=WORLD;

	objectp->handler.attribute=0;
	objectp->handler.coord2=&objectp->co_system;

	return(objectp);
}


// Function 		: InitRefView()
// Coded by		: Scott Evans
// Created/Modified	: 13/11/97

// Description		: Initialise a view (reference type)

// Parameters		: vx,vy,vz - viewpoint coordinates
//				  rx,ry,rz - reference point coordinates
//				  z - view angle

// Returns		: A pointer to a view

// Notes			: MAX_VIEWS sets maximum number of views that can be used,
//				  views are "recycled"


#define MAX_VIEWS 1

GsRVIEW2 *InitRefView(int vx,int vy,int vz,int rx,int ry,int rz,int z)
{
	static GsRVIEW2 views[MAX_VIEWS];
	static int view_no=0;
	GsRVIEW2 *view=&views[view_no=++view_no%MAX_VIEWS];

	// Set view position

	view->vpx=vx;
	view->vpy=vy;
	view->vpz=vz;

	// Set reference position

	view->vrx=rx;
	view->vry=ry;
	view->vrz=rz;

	view->rz=z;

	view->super=WORLD;

	return(view);
}


// Function 		: InitLights()
// Coded by		: Scott Evans
// Created/Modified	: 13/11/97

// Description		: Initialise the 3 flat light sources

// Parameters		: angles - angles for 3 lights
//				  colours - colours for 3 lights

// Returns	     : Pointer to array of 3 lights

// Notes			: N/A


GsF_LIGHT *InitLights(SVECTOR *angles,CVECTOR *colours)
{
	static GsF_LIGHT lights[3]; 
	int i;

	// Set up all 3 light sources position and colours

	for(i=0;i<3;i++)
	{
		lights[i].vx=angles[i].vx;
		lights[i].vy=angles[i].vy;
		lights[i].vz=angles[i].vz;

		lights[i].r=colours[i].r;
		lights[i].g=colours[i].g;
		lights[i].b=colours[i].b;

		GsSetFlatLight(i,&lights[i]);
	}
}

// Function 		: InitFont()
// Coded by		: Scott Evans
// Created/Modified	: 13/11/97

// Description		: Initialise the built in debugging font

// Parameters		: tx,ty - texture page location for font image data

// Returns		: N/A

// Notes			: N/A

void InitFont(long tx,long ty)
{
	FntLoad(tx,ty);
	FntOpen(-(SCRNW>>1)+SCRN_XOFF,-(SCRNH>>1)+SCRN_YOFF,SCRNW,SCRNH,0,500);
}

// Function 		: ControlObject()
// Coded by		: Scott Evans
// Created/Modified	: 13/11/97

// Description		: Control the object with the controller

// Parameters		: object - pointer to TMD object
//				  pad - controller status

// Returns		: N/A

// Notes			: Use controller one to move/rotate object

void ControlObject(TMD_OBJECT *object,unsigned long pad)
{
	MATRIX m;
	
	if(pad&PADRup)
		object->angle.vx-=10;
	if(pad&PADRdown)
		object->angle.vx+=10;
	if(pad&PADRleft)
		object->angle.vy-=10;
	if(pad&PADRright)
		object->angle.vy+=10;
	
	if(pad&PADL1)
		object->angle.vz-=10;
	if(pad&PADR1)
		object->angle.vz+=10;

	if(pad&PADLup)
		object->pos.vy-=10;
	if(pad&PADLdown)
		object->pos.vy+=10;
	if(pad&PADLleft)
		object->pos.vx-=10;
	if(pad&PADLright)
		object->pos.vx+=10;

	if(pad&PADL2)
		object->pos.vz-=10;
	if(pad&PADR2)
		object->pos.vz+=10;

	RotMatrix(&object->angle,&m);
	TransMatrix(&m,&object->pos);
	object->co_system.coord=m;
	object->co_system.flg=0;

	object->angle.vx&=4095;
	object->angle.vy&=4095;
	object->angle.vz&=4095;

	FntPrint("OP : X=%5d Y=%5d Z=%5d\n",object->pos.vx,object->pos.vy,object->pos.vz);
	FntPrint("OA : X=%5d Y=%5d Z=%5d\n",object->angle.vx,object->angle.vy,object->angle.vz);
}

// Function 		: ControlView()
// Coded by		: Scott Evans
// Created/Modified	: 13/11/97

// Description		: Control the view with the controller

// Parameters		: object - pointer to TMD object
//				  pad - controller status

// Returns		: N/A

// Notes			: Use controller two to move the viewpoint/reference point

void ControlView(GsRVIEW2 *view,unsigned long pad)
{
	if(pad&PADRup)
		view->vpy-=5;
	if(pad&PADRdown)
		view->vpy+=5;
	if(pad&PADRleft)
		view->vpx-=5;
	if(pad&PADRright)
		view->vpx+=5;
	
	if(pad&PADL1)
		view->vpz-=5;
	if(pad&PADR1)
		view->vpz+=5;

	if(pad&PADLup)
		view->vry-=5;
	if(pad&PADLdown)
		view->vry+=5;
	if(pad&PADLleft)
		view->vrx-=5;
	if(pad&PADLright)
		view->vrx+=5;

	if(pad&PADL2)
		view->vrz-=5;
	if(pad&PADR2)
		view->vrz+=5;

	if(pad&PADstart)
		view->rz-=ONE;
	if(pad&PADselect)
		view->rz+=ONE;

	FntPrint("VP : X=%5d Y=%5d Z=%5d\n",view->vpx,view->vpy,view->vpz);
	FntPrint("VR : X=%5d Y=%5d Z=%5d\n",view->vrx,view->vry,view->vrz);
}

#ifdef YAROZE

// Function		: PadInit()
// Coded by		: Scott Evans
// Created/Modified	: 16/9/97
// Description		: Initialise controllers

// Parameters		: mode - does nothing

// Returns		: Nothing

// Notes			: Replicates professional development kit version


void PadInit(unsigned short mode)
{
	GetPadBuf(&pad0,&pad1);
}


// Function		: PadRead()
// Coded by		: Scott Evans
// Created/Modified	: 16/9/97
// Description		: Reads controller pads

// Parameters		: id - does nothing

// Returns		: Status of pad0 in lower word, pad1 in upper word 

// Notes			: Replicates professional development kit version

#define _peekByte(addr)	(*(unsigned char *)(addr))

unsigned long PadRead(unsigned short id)
{	
	return(~((_peekByte(pad0+3)|(_peekByte(pad0+2)<<8))|(_peekByte(pad1+3)|(_peekByte(pad1+2)<<8))<<16));
}

#endif
